#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTest.VAL;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion

namespace gov.va.med.vbecs.BOL.UnitTests
{

	
	

	[TestFixture]
	public class BOL_IssuedUnit : BaseTest 
	{
		private BOL.IssuedUnit _tstIU;
		private Guid _bloodUnitGuid;
		private Guid _issuedGuid;
		private byte[] _unitStatusRowVersion;
		private Guid _patientGuid;
		private Guid _busGuid;

		//IU.BloodUnitGuid, IU.IssuedUnitGuid
		private  string issuedUnitQuery = "SELECT TOP 1 VW.BloodUnitStatusGuid, IU.*, PT.PatientGuid, VW.UnitStatusRowVersion FROM dbo.IssuedUnit IU " +
				"JOIN dbo.OrderedUnit OU on IU.BloodUnitGUID = OU.BloodUnitGUID " +
				"JOIN dbo.OrderedComponent OC on OU.OrderedComponentGUID = OC.OrderedComponentGUID " +
				"JOIN dbo.PatientOrder PO on OC.PatientOrderGUID = PO.PatientOrderGUID " +
				"JOIN dbo.PatientTreatment PT on PO.PatientTreatmentGUID = PT.PatientTreatmentGUID " +
				"JOIN dbo.vw_BloodShipAllCurrent VW ON IU.BloodUnitGuid = VW.BloodUnitGuid WHERE " +
				"VW.DivisionCode = '" + UnitTestConfig.TestDivisionCode + 
				"' AND VW.UnitStatusCode IN ('I') " + 
				"AND IU.ReturnDateTime IS NULL AND IU.PatientGuid IS NOT NULL ORDER BY VW.LastUpdateDate DESC";

		private string orderedUnitQuery = "SELECT TOP 1 VW.BloodUnitStatusGuid, OU.OrderedUnitGuid, OU.RowVersion, VW.UnitStatusRowVersion FROM OrderedUnit OU " +
			"JOIN vw_BloodShipAllCurrent VW ON OU.BloodUnitGUID = VW.BloodUnitGUID WHERE VW.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'";

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
				//Create an issued BloodUnit for data
				DataCreator.CreateIssuedBloodUnit(false);
			
				DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable(issuedUnitQuery, true);

				this._bloodUnitGuid = (System.Guid)dtTemp.Rows[0][TABLES.BloodUnit.BloodUnitGuid];
				this._issuedGuid = (System.Guid)dtTemp.Rows[0][TABLES.IssuedUnit.IssuedUnitGuid];
				this._unitStatusRowVersion	= (byte[]) dtTemp.Rows[0][ARTIFICIAL.UnitStatusRowVersion];
				this._busGuid = (System.Guid)dtTemp.Rows[0][TABLES.BloodUnitStatus.BloodUnitStatusGuid];
				this._patientGuid = (System.Guid)dtTemp.Rows[0][TABLES.Patient.PatientGuid];

				this.RefreshData = false;
			}

			_tstIU = new BOL.IssuedUnit();
		}

		#region constructors

		[Test]
		public void Ctor_Blank_Pass()
		{
			BOL.IssuedUnit testObj = new BOL.IssuedUnit();

			Assert.IsTrue(testObj.IssueTechId == string.Empty);
			Assert.IsTrue(testObj.IssueToLocationName == string.Empty);
			Assert.IsTrue(testObj.IssueDate == DateTime.MinValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Blank_Fail()
		{
		}

		[Test]
		public void Ctor_Guid_Pass()
		{
			BOL.IssuedUnit testObj = new BOL.IssuedUnit(_issuedGuid);

			Assert.AreEqual(_issuedGuid, testObj.IssuedUnitGuid, "IssuedUnitGuid");
		}

        [Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Guid_Fail()
		{
		}

		#endregion


		#region methods

		[Test]
		public void GetIssuedUnitsForPatient_Pass()
		{
			DataCreator.CreateIssuedBloodUnit(true);
			DataRow drpatient = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 PatientGuid FROM IssuedUnit WHERE PatientGuid IS NOT NULL ORDER BY LastUpdateDate DESC");
			System.Guid patGuid = (System.Guid)drpatient["PatientGuid"];
			ArrayList units = BOL.IssuedUnit.GetIssuedUnitsForPatient(patGuid);
			Assert.IsTrue(units.Count > 0);
		}
		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetIssuedUnitsForPatient_Fail()
		{
			BOL.IssuedUnit.GetIssuedUnitsForPatient(Guid.Empty);
		}
		[Test]
		public void GetIssuedUnitsForPatient_Fail2()
		{
			ArrayList units = BOL.IssuedUnit.GetIssuedUnitsForPatient(Guid.NewGuid());
			Assert.IsTrue(units.Count == 0);
		}

		[Test]
		public void GetIssuedUnitsByBloodUnitGuid_Pass()
		{
			ArrayList bloodUnitGuids = new ArrayList();
			bloodUnitGuids.Add((Guid) _bloodUnitGuid);
			ArrayList issuedUnits = BOL.IssuedUnit.GetIssuedUnitsByBloodUnitGuid(bloodUnitGuids);
			Assert.IsTrue(issuedUnits.Count > 0);
		}
		[Test]
		public void GetIssuedUnitsByBloodUnitGuid_Fail()
		{
			ArrayList bloodUnitGuids = new ArrayList();
			ArrayList issuedUnits = BOL.IssuedUnit.GetIssuedUnitsByBloodUnitGuid(bloodUnitGuids);
			Assert.IsTrue(issuedUnits.Count == 0);
		}

		[Test]
		public void GetIssuedUnits_Pass()
		{
			DataTable units = BOL.IssuedUnit.GetIssuedUnits();
			Assert.IsTrue(units != null && units.Rows.Count >0);
		}
		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetIssuedUnits_Fail()
		{
		}

		[Test]
		public void GetLocations_Pass()
		{
			VistALinkUnitTestHarness.RequireConnection();
			Assert.IsTrue(_tstIU.GetLocations().Count > 0);
			VistALinkUnitTestHarness.ReleaseConnection();
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetLocations_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			// ToDo: This currently contains all of the fields used in UC69.  There are other
			// fields in the IssuedUnit table that may need to be added here.

			Random r = new Random();
			DateTime dateTime = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			DataRow drIssuedUnit	= GetIssuedUnitDataTable().NewRow();
			
			BOL.IssuedUnit iU = new BOL.IssuedUnit();
			iU.BloodUnitGuid	= _bloodUnitGuid;
			iU.InPatientLocationIndicator = true;
			iU.IssueDate		= dateTime;
			iU.SatisfactoryInspectionIndicator = true;
			iU.IssueTechId		= r.Next(1000).ToString();
			iU.IssueToLocationName = r.Next(1000).ToString();
			iU.IssueToLocationIEN = r.Next(1000);
			iU.LastUpdateUser	= r.Next(1000).ToString() + " User";
			iU.RemoteStorageIndicator = true;
			iU.RemoteStorageText = r.Next(1000).ToString();
			iU.RequestedBy		= r.Next(1000).ToString() + " MD";
			iU.TransporterName	= r.Next(1000).ToString();
	
			iU.LoadDataRowFromThis(drIssuedUnit);

			Assert.IsTrue((Guid) drIssuedUnit[TABLES.IssuedUnit.BloodUnitGuid] == iU.BloodUnitGuid, "BloodUnitGuid");
			Assert.IsTrue((bool) drIssuedUnit[TABLES.IssuedUnit.InPatientLocationIndicator] == iU.InPatientLocationIndicator, "InPatientLocationIndicator");
			Assert.IsTrue((DateTime) drIssuedUnit[TABLES.IssuedUnit.IssueDateTime] == iU.IssueDate,  "IssueDate");
			Assert.IsTrue((bool) drIssuedUnit[TABLES.IssuedUnit.IssueSatisfactoryIndicator] == iU.SatisfactoryInspectionIndicator, "IssueSatisfactory");
			Assert.IsTrue(drIssuedUnit[TABLES.IssuedUnit.IssueTechId].ToString() == iU.IssueTechId, "IssueTechID");
			Assert.IsTrue(drIssuedUnit[TABLES.IssuedUnit.IssueToLocationName].ToString() == iU.IssueToLocationName, "IssueToLocationName");
			//Assertion.IsTrue("Last update function id", (int) drIssuedUnit[TABLES.IssuedUnit.LastUpdateFunctionId] == iU.Las);
			Assert.IsTrue(drIssuedUnit[TABLES.IssuedUnit.LastUpdateUser].ToString() == iU.LastUpdateUser, "LastUpdateUser");
			Assert.IsTrue((bool) drIssuedUnit[TABLES.IssuedUnit.RemoteStorageIndicator] == iU.RemoteStorageIndicator, "RemoteStorageIndicator");
			Assert.IsTrue(drIssuedUnit[TABLES.IssuedUnit.RemoteStorageText].ToString() == iU.RemoteStorageText, "RemoteStorageText");
			Assert.IsTrue(drIssuedUnit[TABLES.IssuedUnit.PhysicianName].ToString() == iU.RequestedBy, "PhysicianName");
			Assert.IsTrue(drIssuedUnit[TABLES.IssuedUnit.TransporterName].ToString() == iU.TransporterName, "TransporterName");
			Assert.IsTrue((int) drIssuedUnit[TABLES.IssuedUnit.LocationIen] == iU.IssueToLocationIEN, "LocationIEN");
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}


		[Test]
		public void Save_Pass()
		{
			this.RefreshData = true;

			DataTable dtIssuedUnits = SprocHelper.GetInsertIssuedUnitSprocTable(System.Guid.NewGuid(), this._bloodUnitGuid, this._patientGuid);
			DataTable dtStatusUnits = SprocHelper.GetUpdateBloodUnitStatusSprocTable(this._busGuid, this._unitStatusRowVersion,false);

			Assert.IsTrue( BOL.IssuedUnit.Save(dtIssuedUnits, null, dtStatusUnits, null, null, null, null,null) );
		}
		[Test]
		[ExpectedException(typeof(Common.RowVersionException))]
		public void Save_Fail()
		{
			DataTable dtIssuedUnits = SprocHelper.GetInsertIssuedUnitSprocTable(System.Guid.NewGuid(), this._bloodUnitGuid, this._patientGuid);
			DataTable dtStatusUnits = SprocHelper.GetUpdateBloodUnitStatusSprocTable(this._busGuid, this._unitStatusRowVersion, false);

			dtStatusUnits.Rows[0][TABLES.BloodUnitStatus.RowVersion] = new byte[8];

			BOL.IssuedUnit.Save(dtIssuedUnits, null, dtStatusUnits, null, null, null, null,null);
		}

		[Test]
		public void Update_Pass()
		{
			DataTable dtProduct = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM BloodProduct WHERE ProductTypeCode = 'E002' AND RecordStatusCode = 'A' ORDER BY LastUpdateDate DESC");
			
			System.Guid bloodUnitGuid = System.Guid.NewGuid();
			System.Guid bloodProductGuid = (System.Guid)dtProduct.Rows[0][TABLES.BloodProduct.BloodProductGuid];
			bool isCodabar = (dtProduct.Rows[0][TABLES.BloodProduct.ProductCodeIndicator].ToString() == "0");
			DataCreator.CreateBloodUnit(bloodUnitGuid, System.Guid.NewGuid(), Guid.Empty, bloodProductGuid, isCodabar);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(bloodUnitGuid, bloodProductGuid);
			//
			Guid issuedUnitGuid = Guid.NewGuid();
			DataTable dtIssuedUnits = SprocHelper.GetInsertIssuedUnitSprocTable(issuedUnitGuid, bloodUnitGuid, this._patientGuid);
			DataTable dtStatusUnits = SprocHelper.GetUpdateBloodUnitStatusSprocTable(tmpBU.BloodUnitStatus.BloodUnitStatusGuid, tmpBU.BloodUnitStatus.RowVersion,false);
			//
			BOL.IssuedUnit.Save(dtIssuedUnits, null, dtStatusUnits, null, null, null, null,null);
			//
			BOL.IssuedUnit returnIU = new BOL.IssuedUnit(issuedUnitGuid);
			returnIU.ReturnDateTime = DateTime.Now;
			returnIU.ReturnTechId = returnIU.IssueTechId;
			returnIU.ReturnFromLocationName = returnIU.IssueToLocationName;
			ArrayList returnUnits = new ArrayList(1);
			returnUnits.Add(returnIU);
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.ReleaseUnitFromPatientBackToInventory);
			bool ret = BOL.IssuedUnit.Update(returnUnits,new ArrayList(),Common.WorkloadProcessID.ReleaseUnitFromPatientBackToInventory, this._patientGuid);
			//
			Assert.AreEqual(true,ret,"Update");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(NullReferenceException))]
		public void Update_Fail()
		{
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.IssueUnit);
			BOL.IssuedUnit.Update(null, null, Common.WorkloadProcessID.IssueUnit,this._patientGuid);
		}

		[Test]
		public void MarkUnitUnsatisfactory_Pass()
		{
			this.RefreshData = true;

			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable(orderedUnitQuery, true);

			Guid orderedUnitGuid		= (Guid)dtTemp.Rows[0][TABLES.OrderedUnit.OrderedUnitGuid];
			byte[] rowVersion			= (byte[])dtTemp.Rows[0][TABLES.OrderedUnit.RowVersion];
			byte[] unitStatusRowVersion	= (byte[])dtTemp.Rows[0]["UnitStatusRowVersion"];

			Assert.IsTrue(BOL.IssuedUnit.MarkUnitUnsatisfactory(orderedUnitGuid, rowVersion, unitStatusRowVersion, Common.UpdateFunction.UnitTests,null));
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void MarkUnitUnsatisfactory_Fail()
		{
			BOL.IssuedUnit.MarkUnitUnsatisfactory(Guid.Empty, null, null, Common.UpdateFunction.UnitTests,null);
		}

		[Test]
		public void GetIssuedUnitFromBloodUnitGuid_Pass()
		{
			BOL.IssuedUnit tmpIssuedUnit=BOL.IssuedUnit.GetIssuedUnitFromBloodUnitGuid(this._bloodUnitGuid);

			Assert.IsNotNull(tmpIssuedUnit);
			Assert.IsTrue(!tmpIssuedUnit.IsDirty);
		}

		[Test]
		[ExpectedException(typeof(System.IndexOutOfRangeException))]
		public void GetIssuedUnitFromBloodUnitGuid_Fail()
		{
			BOL.IssuedUnit tmpIssuedUnit=BOL.IssuedUnit.GetIssuedUnitFromBloodUnitGuid(System.Guid.NewGuid());

			Assert.IsNotNull(tmpIssuedUnit);
		}

		[Test]
		public void GetIssuedToLocationList_Pass()
		{
			Assert.IsNotNull(IssuedUnit.GetIssueToLocationList());
			Assert.IsTrue(IssuedUnit.GetIssueToLocationList().Count>0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetIssuedToLocationList_Fail()
		{
		}

		[Test]
		public void GetPhysiciansList_Pass()
		{
			ArrayList list = IssuedUnit.GetPhysiciansList();
			Assert.IsNotNull( list );
			Assert.IsTrue( list.Count > 0 );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetPhysiciansList_Fail()
		{
		}

		[Test]
		public void GetRemoteStorageLocations_Pass()
		{
			BOL.BloodUnit tmpUnit = DataCreator.CreateRBCBloodUnit(true);
			DataCreator.CreatePatientAssignment(tmpUnit.BloodUnitGuid, Guid.NewGuid(), true, true);
			DataCreator.InsertIssuedUnit(Guid.NewGuid(), tmpUnit.BloodUnitGuid);
			ArrayList list = IssuedUnit.GetRemoteStorageLocations();
			Assert.IsNotNull( list );
			Assert.IsTrue( list.Count > 0 );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetRemoteStorageLocations_Fail()
		{
		}

		[Test]
		public void MarkUnitUnsatisfactory_3_Par_Pass()
		{
            DataTable dtPatient = DataCreator.GetRandomPatientRecord(true);
			Guid patientGuid = Guid.Empty;
			if (dtPatient.Rows.Count > 0)
			{
				patientGuid = (Guid)dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			}
			else
			{
				Assert.AreEqual(patientGuid, Guid.NewGuid(), "Patient does not exist");
				return;
			}
			//
			Guid bloodUnitGuid = Guid.NewGuid();
			//
			DataTable dtProduct = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM BloodProduct WHERE ProductTypeCode = 'E002' AND RecordStatusCode = 'A' ORDER BY LastUpdateDate DESC");
			Guid bloodProductGuid = (Guid)dtProduct.Rows[0][TABLES.BloodProduct.BloodProductGuid];
			bool isCodabar = (dtProduct.Rows[0][TABLES.BloodProduct.ProductCodeIndicator].ToString() == "0");
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), patientGuid, bloodProductGuid, isCodabar);
			DataTable dtUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE BloodunitGuid = '"+ bloodUnitGuid.ToString()+"'");
			//
			Guid patientTreatmentGuid = Guid.NewGuid();
			string insertPatientTreatment = "INSERT INTO PATIENTTREATMENT (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Environment.UserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientOrderGuid = Guid.NewGuid();
			string insertPatientOrder = "INSERT INTO PATIENTORDER (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Environment.UserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedComponentGuid = Guid.NewGuid();
			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedComponent += "'1',";
            insertOrderedComponent += "'" + SprocHelper.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += "'2',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Environment.UserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedUnitGuid = Guid.NewGuid();
			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid bloodUnitTestGuid1 = Guid.NewGuid();
			string insertBloodUnitTest1 = "INSERT INTO BLOODUNITTEST (";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.BloodUnitTestGuid + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.BloodUnitGuid + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.BloodTestTypeId + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.TestDate + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.TestTechId + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.TestResultId + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.EntryTechId + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.EntryMethodCode + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.OrderedComponentGuid + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.OrderedUnitGuid + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.ReactivityPhaseCode + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.RecordStatusCode + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.DivisionCode + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.LastUpdateDate + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.LastUpdateUser + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.LastUpdateFunctionId + ")";
			insertBloodUnitTest1 += "VALUES (";
			insertBloodUnitTest1 += "'" + bloodUnitTestGuid1.ToString() + "',";
			insertBloodUnitTest1 += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitTest1 += "'100',";
			insertBloodUnitTest1 += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest1 += "'" + Environment.UserName + "',";
			insertBloodUnitTest1 += "'0',";
			insertBloodUnitTest1 += "'" + Environment.UserName + "',";
			insertBloodUnitTest1 += "'M',";
			insertBloodUnitTest1 += "'" + orderedComponentGuid.ToString() + "',";
			insertBloodUnitTest1 += "'" + orderedUnitGuid.ToString() + "',";
			insertBloodUnitTest1 += "'2',";
			insertBloodUnitTest1 += "'A',";
			insertBloodUnitTest1 += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertBloodUnitTest1 += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest1 += "'" + Environment.UserName + "',";
			insertBloodUnitTest1 += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid bloodUnitTestGuid2 = Guid.NewGuid();
			string insertBloodUnitTest2 = "INSERT INTO BLOODUNITTEST (";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.BloodUnitTestGuid + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.BloodUnitGuid + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.BloodTestTypeId + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.TestDate + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.TestTechId + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.TestResultId + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.EntryTechId + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.EntryMethodCode + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.OrderedComponentGuid + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.OrderedUnitGuid + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.ReactivityPhaseCode + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.RecordStatusCode + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.DivisionCode + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.LastUpdateDate + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.LastUpdateUser + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.LastUpdateFunctionId + ")";
			insertBloodUnitTest2 += "VALUES (";
			insertBloodUnitTest2 += "'" + bloodUnitTestGuid2.ToString() + "',";
			insertBloodUnitTest2 += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitTest2 += "'102',";
			insertBloodUnitTest2 += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest2 += "'" + Environment.UserName + "',";
			insertBloodUnitTest2 += "'C',";
			insertBloodUnitTest2 += "'" + Environment.UserName + "',";
			insertBloodUnitTest2 += "'M',";
			insertBloodUnitTest2 += "'" + orderedComponentGuid.ToString() + "',";
			insertBloodUnitTest2 += "'" + orderedUnitGuid.ToString() + "',";
			insertBloodUnitTest2 += "'2',";
			insertBloodUnitTest2 += "'A',";
			insertBloodUnitTest2 += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertBloodUnitTest2 += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest2 += "'" + Environment.UserName + "',";
			insertBloodUnitTest2 += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid issuedUnitGuid = Guid.NewGuid();
			string insertIssuedUnit = "INSERT INTO ISSUEDUNIT (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedComponent);
			UnitTestUtil.RunSQL(insertOrderedUnit);
			UnitTestUtil.RunSQL(insertBloodUnitTest1);
			UnitTestUtil.RunSQL(insertBloodUnitTest2);
			UnitTestUtil.RunSQL(insertIssuedUnit);

			BOL.IssuedUnit iu = new IssuedUnit( issuedUnitGuid );
			BOL.OrderedUnit ou = new OrderedUnit(bloodUnitGuid, patientGuid);
			iu.OrderedUnitGuid = ou.OrderedUnitGuid;
			iu.OrderedUnitRowVersion = ou.RowVersion;
			iu.ReturnDateTime = DateTime.Now;
			iu.ReturnTechId = Environment.UserName;
			iu.BloodUnitGuid = bloodUnitGuid;
			iu.PatientGuid = patientGuid;
			//
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.IssueUnit);
			bool results = IssuedUnit.MarkUnitUnsatisfactory( iu, new ArrayList(), Common.WorkloadProcessID.IssueUnit );
			//
			Assert.IsTrue( results );
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void MarkUnitUnsatisfactory_3_Par_Fail()
		{
			bool results = IssuedUnit.MarkUnitUnsatisfactory( null, null, Common.WorkloadProcessID.Unknown );
			//
			Assert.IsTrue( !results );
		}

		#endregion


		#region properties

		[Test]
		public void IssuedUnitGuid_Pass()
		{
			Guid issuedUnitGuid = Guid.NewGuid();
			_tstIU.IssuedUnitGuid = issuedUnitGuid;
			Assert.IsTrue(_tstIU.IssuedUnitGuid == issuedUnitGuid);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnitGuid_Fail()
		{
		}

		[Test]
		public void BloodUnitGuid_Pass()
		{
			Guid bloodUnitGuid = Guid.NewGuid();
			_tstIU.BloodUnitGuid = bloodUnitGuid;
			Assert.IsTrue(_tstIU.BloodUnitGuid == bloodUnitGuid);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BloodUnitGuid_Fail()
		{
		}

		[Test]
		public void RequestedBy_Pass()
		{
			Random r = new Random();
			string requestedBy = r.Next(1000).ToString();
			_tstIU.RequestedBy = requestedBy;
			Assert.IsTrue(_tstIU.RequestedBy == requestedBy);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RequestedBy_Fail()
		{
		}

		[Test]
		public void IssueDate_Pass()
		{
			DateTime issueDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			_tstIU.IssueDate = issueDate;
			Assert.IsTrue(_tstIU.IssueDate == issueDate);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssueDate_Fail()
		{
		}

		[Test]
		public void IssueTechId_Pass()
		{
			Random r = new Random();
			string issueTechId = r.Next(1000).ToString();
			_tstIU.IssueTechId = issueTechId;
			Assert.IsTrue(_tstIU.IssueTechId == issueTechId);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssueTechId_Fail()
		{
		}

		[Test]
		public void TransporterName_Pass()
		{
			Random r = new Random();
			string transporterName = r.Next(1000).ToString();
			_tstIU.TransporterName = transporterName;
			Assert.IsTrue(_tstIU.TransporterName == transporterName);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void TransporterName_Fail()
		{
		}

		[Test]
		public void IssueToLocationName_Pass()
		{
			Random r = new Random();
			string issueToLocationName = r.Next(1000).ToString();
			_tstIU.IssueToLocationName = issueToLocationName;
			Assert.IsTrue(_tstIU.IssueToLocationName == issueToLocationName);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssueToLocationName_Fail()
		{
		}

		[Test]
		public void RemoteStorageIndicator_Pass()
		{		
			_tstIU.RemoteStorageIndicator = true;
			Assert.IsTrue(_tstIU.RemoteStorageIndicator == true);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RemoteStorageIndicator_Fail()
		{
		}

		[Test]
		public void RemoteStorageText_Pass()
		{
			Random r = new Random();
			string remoteStorageText = r.Next(1000).ToString();
			_tstIU.RemoteStorageText = remoteStorageText;
			Assert.IsTrue(_tstIU.RemoteStorageText == remoteStorageText);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RemoteStorageText_Fail()
		{
		}

		[Test]
		public void InPatientLocationIndicator_Pass()
		{
			_tstIU.InPatientLocationIndicator = true;
			Assert.IsTrue(_tstIU.InPatientLocationIndicator == true);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void InPatientLocationIndicator_Fail()
		{
		}

		[Test]
		public void SatisfactoryInspectionIndicator_Pass()
		{
			_tstIU.SatisfactoryInspectionIndicator = true;
			Assert.IsTrue(_tstIU.SatisfactoryInspectionIndicator == true);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SatisfactoryInspectionIndicator_Fail()
		{
		}

		[Test]
		public void RecordStatusCode_Pass()
		{
			_tstIU.RecordStatusCode = Common.RecordStatusCode.Active;
			Assert.IsTrue(_tstIU.RecordStatusCode == Common.RecordStatusCode.Active);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RecordStatusCode_Fail()
		{
		}

		[Test]
		public void IssueToLocationIEN_Pass()
		{
			Random r = new Random();
			int issueToLocationIEN = r.Next(1000);
			_tstIU.IssueToLocationIEN = issueToLocationIEN;
			Assert.IsTrue(_tstIU.IssueToLocationIEN == issueToLocationIEN);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssueToLocationIEN_Fail()
		{
		}

		[Test]
		public void LastUpdateDate_Pass()
		{
			DateTime dateTime = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			_tstIU.LastUpdateDate = dateTime;
			Assert.IsTrue(_tstIU.LastUpdateDate == dateTime);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LastUpdateDate_Fail()
		{
		}

		[Test]
		public void LastUpdateUser_Pass()
		{
			Random r = new Random();
			string lastUpdateUser = r.Next(1000).ToString();
			_tstIU.LastUpdateUser = lastUpdateUser;
			Assert.IsTrue(_tstIU.LastUpdateUser == lastUpdateUser);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LastUpdateUser_Fail()
		{
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RowVersion_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RowVersion_Fail()
		{
		}

		[Test]
		public void ReturnSatisfactoryIndicator_Pass()
		{
			_tstIU.ReturnSatisfactoryIndicator = true;
			Assert.IsTrue(_tstIU.ReturnSatisfactoryIndicator == true);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ReturnSatisfactoryIndicator_Fail()
		{
		}

		[Test]
		public void ReturnFromLocationName_Pass()
		{
			Random r = new Random();
			string returnFromLocationName = r.Next(1000).ToString();
			_tstIU.ReturnFromLocationName = returnFromLocationName;
			Assert.IsTrue(_tstIU.ReturnFromLocationName == returnFromLocationName);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ReturnFromLocationName_Fail()
		{
		}

		[Test]
		public void ReturnTechId_Pass()
		{
			Random r = new Random();
			string returnTechId = r.Next(1000).ToString();
			_tstIU.ReturnTechId = returnTechId;
			Assert.IsTrue(_tstIU.ReturnTechId == returnTechId);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ReturnTechId_Fail()
		{
		}

		[Test]
		public void ReturnDateTime_Pass()
		{
			DateTime dateTime = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			_tstIU.ReturnDateTime = dateTime;
			Assert.IsTrue(_tstIU.ReturnDateTime == dateTime);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ReturnDateTime_Fail()
		{
		}

		[Test]
		public void SpecimenExpirationDate_Pass()
		{
			DateTime dateTime = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			_tstIU.SpecimenExpirationDate = dateTime;
			Assert.IsTrue(_tstIU.SpecimenExpirationDate == dateTime);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SpecimenExpirationDate_Fail()
		{
		}

		[Test]
		public void OrderedUnitRowVersion_Pass()
		{
			Random r = new Random();
			byte[] rowVersion = new byte[8];
			r.NextBytes(rowVersion);
			_tstIU.OrderedUnitRowVersion = rowVersion;
			Assert.IsTrue(_tstIU.OrderedUnitRowVersion == rowVersion);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void OrderedUnitRowVersion_Fail()
		{
		}

		[Test]
		public void OrderedUnitGuid_Pass()
		{
			Guid orderedUnitGuid = Guid.NewGuid();
			_tstIU.OrderedUnitGuid = orderedUnitGuid;
			Assert.IsTrue(_tstIU.OrderedUnitGuid == orderedUnitGuid);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void OrderedUnitGuid_Fail()
		{
		}

		[Test]
		public void PatientGuid_Pass()
		{
			Guid patientGuid = Guid.NewGuid();
			_tstIU.PatientGuid = patientGuid;
			Assert.IsTrue(_tstIU.PatientGuid == patientGuid);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PatientGuid_Fail()
		{
		}

		[Test]
		public void OutOfUncontrolledStorage_Pass()
		{
			_tstIU.RemoteStorageIndicator = true;
			Assert.IsTrue(!_tstIU.OutOfUncontrolledStorage);
		}
		[Test]
		public void OutOfUncontrolledStorage_Fail()
		{
			_tstIU.RemoteStorageIndicator = false;
			_tstIU.ReturnDateTime = _tstIU.IssueDate.AddHours(1);
			Assert.IsTrue(_tstIU.OutOfUncontrolledStorage);
		}

		#endregion


		#region helper methods

		public static DataTable GetIssuedUnitDataTable()
		{
			DataTable dtIssueUnits = new DataTable();
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.BloodUnitGuid, typeof(Guid));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.InPatientLocationIndicator, typeof(bool));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.IssueDateTime, typeof(DateTime));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.IssuedUnitGuid, typeof(Guid));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.IssueSatisfactoryIndicator, typeof(bool));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.IssueTechId, typeof(string));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.IssueToLocationName, typeof(string));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.LastUpdateDate, typeof(DateTime));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.LastUpdateFunctionId, typeof(int));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.LastUpdateUser, typeof(string));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.RemoteStorageIndicator, typeof(bool));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.RemoteStorageText, typeof(string));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.PhysicianName, typeof(string));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.TransporterName, typeof(string));
			dtIssueUnits.Columns.Add(TABLES.IssuedUnit.LocationIen, typeof(int));
			dtIssueUnits.Columns.Add(TABLES.VamcDivision.DivisionCode, typeof(string));
			return dtIssueUnits;
		}

		[Test]
		public void SetWarningMessages_Pass()
		{
			DataRow dtRow = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 BloodUnitGuid FROM BloodUnitMedia WHERE UnitExpirationDate > '" + DateTime.Now.ToString() + "'");
			BOL.IssuedUnit _iu = new BOL.IssuedUnit(Guid.NewGuid());
			_iu.BloodUnitGuid = (Guid) dtRow["BloodUnitGuid"];
			_iu.SetWarningMessages();
			//there should be one messages on the new issued unit for unit expiration 1/1/01 < current date
			Assert.IsTrue(_iu.GetMessagesForTooltip() == String.Empty);
		}

		[Test]
		public void SetWarningMessages_Fail()
		{
			BOL.IssuedUnit _iu = new BOL.IssuedUnit(Guid.NewGuid());
			_iu.SpecimenExpirationDate = DateTime.Now.AddDays(-1);
			_iu.SetWarningMessages();
			//there should be two messages on the new issued unit 
			Assert.IsTrue(_iu.GetMessagesForTooltip() != String.Empty);
		}

		[Test]
		public void GetMessagesForTooltip_Pass()
		{
			BOL.IssuedUnit _iu = new BOL.IssuedUnit(Guid.NewGuid());
			_iu.SetWarningMessages();
			//there should be one messages on the new issued unit for unit expiration 1/1/01 < current date
			Assert.IsTrue(_iu.GetMessagesForTooltip() != String.Empty);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetMessagesForTooltip_Fail()
		{
		}

		
		[Test]
		public void EmergencyIssue_Pass()
		{
			bool tstValue = (!this._tstIU.EmergencyIssue);

			this._tstIU.EmergencyIssue = tstValue;

			Assert.AreEqual(tstValue, this._tstIU.EmergencyIssue);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void EmergencyIssue_Fail()
		{
		}

		#endregion
	}
}

#endif
